# © 2025 Digital Advantage Corp.

######################################################################
#
# lib_common.rb： 汎用的な関数ライブラリ
#
# for Ruby 3.2

######################################################################

# 以下のGemはあらかじめインストールしておくこと
require "socket" # 自身のIPアドレス取得用
require "mail" # Eメール通知用
require "logger" # ログ記録用

######################################################################
#
# loggerによるログの初期設定
#
$logger_device = STDOUT # 出力先をバックアップ
$logger = Logger.new($logger_device) # 標準出力へ
$logger.level = :info # INFOレベル以上を記録
$logger.formatter = proc do |severity, datetime, progname, msg| # 日付を含むフォーマット
	[
		"[#{datetime.strftime('%Y-%m-%d %H:%M:%S.%L')}]", # 日付
		severity.to_s == "INFO" ? "       " : "[" + format("% 5s", severity.to_s) + "]", # ログレベル
		progname && progname != "" ? " #{progname}()：" : "", # メソッド
		" #{msg}\n" # メッセージ
	].join("")
end

######################################################################
#
# 汎用のリトライ処理
#
# times： 最大試行回数
# interval： 試行間隔（秒）
# force_debug： デバッグ出力を強制するかどうか
# exit_code： エラー発生時の終了コード
# 戻り値： true： 正常終了、false： 再試行しても正常に実行できなかった
#
def general_retry(times: 3, interval: 1, force_debug: false, exit_code: EXIT_CODE::GENERAL_FAILURE)

	is_success = false

	try = 0
	begin
		try += 1
		yield
		is_success = true
	rescue => ex # StandardErrorをcatch
		if try < times
			retry_message = "例外にともなう再試行中です： #{try} 回目"
			$logger.error(make_error_msg_from_exception ex, retry_message)
			if force_debug
				$debug = true
				$logger.level = :debug # デバッグ出力を有効化
			end
			sleep(interval)
			interval *= 2 # 2倍ずつ延ばしていく
			retry
		else
			error_message = "#{try} 回試行するも正常には実行できませんでした"
			$logger.fatal(make_error_msg_from_exception ex, error_message)
			is_success = false
		end
	end
	$exit_code = exit_code if !is_success

	is_success
end

######################################################################
#
# プラットフォームがWindowsなら真を返す
#
def is_win?
	RUBY_PLATFORM.downcase.match?(/msys|mingw|cygwin|bccwin|wince|mswin(?!ce)|emc/)
end

######################################################################
#
# 呼び出し元のスクリプト名（パス）を返す
#
def get_my_own_name
	my_own_name = File.expand_path($0)
	my_own_name = my_own_name.encode("UTF-8", "Windows-31J") if is_win?
end

######################################################################
#
# 自身のIPアドレスを返す
#
def get_my_ipv4_address
	udp_socket = UDPSocket.new
	udp_socket.connect("128.0.0.0", 7)
	socket_name = udp_socket.getsockname
	port, ipv4 = Socket.unpack_sockaddr_in(socket_name)
	udp_socket.close

	ipv4
end

######################################################################
#
# Eメールで通知する
#
# subject： メール題目（UTF-8）
# body： 本文（UTF-8）
# 戻り値： 成功したらtrue、失敗したらfalse
#
def send_alert_by_mail subject, body
	is_success = false

	# 電子メールを生成
	mail = Mail.new do
		subject	subject.scrub
		body	body.scrub
		from	$mail_sender
		to		$mail_receiver_pc
		cc		$mail_receiver_mobile
	end

	# メール送信オプション
	options = $send_mail_options

	# メール送信を実行。リトライあり
	is_success = general_retry(times: 3, interval: 3.0) {
		mail.charset = 'utf-8'
		mail.delivery_method(:smtp, options)
		mail.deliver
		$logger.debug(__method__) { "メール送信に成功しました" }
	}
end

######################################################################
#
# 通知時の定例のメッセージを生成する
#
def make_notice_msg
	host_name = `hostname`.strip
	ruby_version = "#{RUBY_VERSION} #{RUBY_PLATFORM}"
	my_own_name = get_my_own_name
	my_own_ipv4 = get_my_ipv4_address

	notice_message = "―――\n"
	notice_message += " 送信元： #{host_name}（#{my_own_ipv4}）\n"
	notice_message += " Rubyのバージョン： #{ruby_version}\n"
	notice_message += " 実行ファイル： #{my_own_name}\n"
	notice_message += "―――"
end

######################################################################
#
# 例外からエラーメッセージを生成する
#
def make_error_msg_from_exception ex, base_msg
	error_message = ""
	error_message += "#{base_msg}\n" if base_msg && base_msg != ""
	error_message += "#{ex.message}\n"
	error_message += "クラス： #{ex.class}\n"
	error_message += "バックトレース：\n"
	error_message += ex.backtrace.join("\n") + "\n";
	error_message += make_notice_msg
end
