# © 2025 Digital Advantage Corp.

######################################################################
#
# AzureサービスのIPアドレス範囲一覧などをまとめたJSONファイルをMicrosoftサイトから取得し、
# NGINXのreal_ipモジュール向け設定ファイルに変換して、App Serviceにアップロードする
#
# for Ruby 3.2

Version = "2025-03-12" # --versionオプションで表示

######################################################################

$debug = false # デバッグ用
$my_own_title = "AzureサービスのIPアドレス範囲一覧JSONの取得とNGINX設定テキストへの変換"

######################################################################

# 以下のGemはあらかじめインストールしておくこと
require "json"
require "optparse" # コマンドラインオプションのパース用
require "open-uri" # HTMLファイルのダウンロード用
require "nokogiri" # HTMLファイルのパース用
require "net/ftp" # 生成したテキストファイルのアップロード用
require "pp" # デバッグ表示用

require_relative "inc/const" # パスやURLなどはここに記載
require_relative "inc/lib_common" # 汎用関数ライブラリ

######################################################################
#
# URLで指定されたファイルをダウンロードする
#
def download_file_by_url downloaded_url, use_proxy = true
	downloaded_file = nil
	proxies = $proxy_servers.map{ |ps| ps ? URI::parse(ps) : nil } # プロキシサーバの選択

	general_retry(times: 10, interval: 9.37) {
		if use_proxy
			proxy = proxies[rand(proxies.size)] # 集中アクセスを防ぐためにランダム化
		else
			proxy = nil
		end

		downloaded_file = URI.open(downloaded_url, :proxy => proxy, "User-Agent" => $crawl_user_agent) { |uri| uri.read } 
	}

	downloaded_file
end

######################################################################
#
# HTMLから対象のJSONファイルのダウンロードURLを取得する
#
def get_json_url_from_downloaded_html downloaded_html, json_regexp
	json_url = json_file_name = nil

	# 取得したhtmlをNokogiriでパースする
	downloaded_doc = Nokogiri::HTML.parse(downloaded_html)
	$logger.debug(__method__) { PP.pp(downloaded_doc, "") }

	downloaded_doc.search("main").each do |main| # mainタグ内
		$logger.debug(__method__) { PP.pp(main, "") }

		main.search("a").each do |a_tag| # aタグ内
			$logger.debug(__method__) { PP.pp(a_tag, "") }

			url = a_tag.attr("href")
			if url.match?(json_regexp)
				json_url = url
				$logger.debug(__method__) { PP.pp(json_url, "") }
				json_file_name = json_url.sub(json_regexp, '\2')
			end
		end
	end

	[ json_url, json_file_name ]
end

######################################################################
#
# 保存してあるURLと現在のURLが違っていたら、真を返す
#
def is_changed_url url, latest_url_file
	is_changed = true # 不一致
	latest_json_url = nil

	if File.exist?(latest_url_file)
		current_url = url.to_s.strip
		general_retry(times: 3, interval: 1.0) {
			latest_json_url = File.open(latest_url_file, "rb") { |file| file.read.to_s.strip }
		}
		if latest_json_url != "" && current_url != "" && current_url === latest_json_url
			# 一致
			is_changed = false
		end
	end

	is_changed
end

######################################################################
#
# JSONのURLを保存する
#
def save_json_url json_url, latest_url_file
	general_retry(times: 3, interval: 1.0) {
		File.open(latest_url_file, "wb") { |file| file.write(json_url) }
	}
end

######################################################################
#
# JSONのURL記録ファイルをクリアする
#
def clear_json_url latest_url_file
	if File.exist?(latest_url_file)
		general_retry(times: 3, interval: 1.0) {
			File.delete(latest_url_file)
		}
	end
end

######################################################################
#
# Azure FrontDoor（AFD）のバックエンドIP一覧を取得する
#
def get_ip_ranges_for_azure_service all_azure_ip_ranges, service_name
	azure_service_ip_ranges = []

	if all_azure_ip_ranges.has_key?("values")
		values = all_azure_ip_ranges['values']
		service = values.select { |c| c['name'] == service_name }
		service = service[0]
		if service && service.size > 0
			if service.has_key?("properties")
				properties = service['properties']
				if properties.has_key?("addressPrefixes")
					azure_service_ip_ranges = properties['addressPrefixes']
				end
			end
		end
	end

	azure_service_ip_ranges
end

######################################################################
#
# Azure FrontDoor（AFD）のバックエンドIP一覧のNGINX real_ipモジュール用設定を生成する
#
def create_nginx_real_ip_for_afd_backend all_azure_ip_ranges
	settings_nginx_real_ip = ""

	service_name = "AzureFrontDoor.Backend"
	afd_backend_ip_ranges = get_ip_ranges_for_azure_service all_azure_ip_ranges, service_name

	afd_backend_ip_ranges.each do |ip_range|
		settings_nginx_real_ip += "set_real_ip_from #{ip_range};" + "\n"
	end

	settings_nginx_real_ip
end

######################################################################
#
# FTP putでアップロード
#
def upload_file_by_ftp site, files
	is_success = false

	hostname = site['hostname']
	username = site['username']
	password = site['password']
	is_ssl = site['is_ssl']
	remote_dir = site['remote_dir']

	$logger.debug(__method__) { "サイトの概要： #{site['desc']}" }

	is_success = general_retry(times: 3, interval: 5.0) {
		Net::FTP.open(
			hostname,
			ssl: is_ssl,
			username: username, 
			password: password,
			debug_mode: false
		) do |ftp|
			ftp.chdir(remote_dir)
			files.each do |f|
				local_file_path = f['local']
				remote_file_name = f['remote']
				ftp.putbinaryfile(local_file_path, remote_file_name)
				$logger.info "FTPで転送しました： #{local_file_path} => #{remote_file_name}"
			end
		end
	}

	is_success
end

######################################################################
#
# コマンドラインオプションのパース
#
def parse_cmd_line_options
	params = {
	}

	opt = OptionParser.new

	# デバッグオプション
	opt.on("-d", "--[no-]debug", "Enable/Disable debug mode.") { |v| $debug = v }

	opt.parse!(ARGV, into: params)

	params
end

######################################################################
#
# メインルーチン
#
def main

	params = parse_cmd_line_options # コマンドラインオプションをハッシュで取得
	$logger.debug(__method__) { PP.pp(params, "") }

	$logger.level = $logger_level = :debug if $debug # デバッグオプションが有効ならログ記録レベルを下げる

	# ダウンロードするWebページ
	downloaded_url = $msdc_page_url
	$logger.debug(__method__) { PP.pp(downloaded_url, "") }

	# 前回ダウンロードしたHTMLファイル
	html_saved_path = $msdc_confirm_file
	$logger.debug(__method__) { PP.pp(html_saved_path, "") }

	# ダウンロードページのHTMLを読み込む
	$logger.info "HTMLを読み込み中： #{downloaded_url}"
	downloaded_html = download_file_by_url downloaded_url
	$logger.debug(__method__) { PP.pp(downloaded_html, "") }
	if downloaded_html
		# 読み込めたHTMLをローカルに保存
		general_retry(times: 3, interval: 1.0) {
			File.open(html_saved_path, "wb") { |file| file.write(downloaded_html) }
		}
	else
		raise "HTMLの読み込みに失敗しました。URL： #{downloaded_url}" 
	end
	$logger.info "HTMLを読み込みました"

	# HTMLからJSONファイルのURLを抽出
	# 例： https://download.microsoft.com/download/7/1/d/71d86715-5596-4529-9b13-da13a5de5b63/ServiceTags_Public_20250303.json
	json_regexp = /^(https:\/\/download\.microsoft\.com\/download\/\w\/\w\/\w\/[\w\-]+\/)([\w\-_]+\.json)$/
	json_url, json_file_name = get_json_url_from_downloaded_html downloaded_html, json_regexp
	raise "JSONファイルURLの抽出に失敗しました。" if !json_url
	$logger.info "HTMLからJSONファイルのURLを抽出しました： #{json_url}"

	# JSONファイルをダウンロード
	$logger.info "JSONファイルをダウンロードしています： #{json_url}"
	json = download_file_by_url json_url
	raise "JSONファイルのダウンロードに失敗しました。" if !json
	$logger.info "JSONファイルをダウンロードしました"

	# JSONファイルを保存
	json_file_path = File.join($data_dir, json_file_name)
	json_file_path = File.expand_path(json_file_path)
	general_retry(times: 3, interval: 1.0) {
		File.open(json_file_path, "wb") { |file| file.write(json) }
	}
	$logger.info "JSONファイルを保存しました: #{json_file_path}"

	# Azure FrontDoorのIP範囲を、NGINXのreal_ipモジュール向け設定テキストに変換する
	$logger.info "JSONファイルからNGINXの設定テキストを生成しています"
	all_azure_ip_ranges = JSON.parse(json)
	nginx_real_ip_for_afd_backend = create_nginx_real_ip_for_afd_backend all_azure_ip_ranges
	$logger.debug(__method__) { PP.pp(nginx_real_ip_for_afd_backend, "") }

	# テキストファイルに保存
	nginx_real_ip_file_path = File.join($data_dir, $nginx_real_ip_file_name)
	nginx_real_ip_file_path = File.expand_path(nginx_real_ip_file_path)
	general_retry(times: 3, interval: 1.0) {
		File.open(nginx_real_ip_file_path, "wb") { |file| file.write nginx_real_ip_for_afd_backend }
	}
	$logger.info "NGINX設定ファイルを保存しました： #{nginx_real_ip_file_path}"

	# NGINX設定ファイルをApp ServiceにFTPでアップロード
	$logger.info "NGINX設定ファイルをFTPでApp Serviceにアップロードします"
	ftp_site = $params_ftp_appsvc
	ftp_files = [
		{ "local" => nginx_real_ip_file_path,	"remote" => $nginx_real_ip_file_name },
	]
	$logger.info "#{nginx_real_ip_file_path} を #{ftp_site['hostname']} へFTPでアップロードしています……"
	is_success_ftp = upload_file_by_ftp ftp_site, ftp_files
	raise "FTPサイトへのアップロードに失敗しました。" if !is_success_ftp
	$logger.info "FTPサイトへのアップロードが完了しました"

	EXIT_CODE::NORMAL_END
end

######################################################################

$logger.info "―――――――――― #{$my_own_title} ――――――――――"

$exit_code = EXIT_CODE::NORMAL_END # シェルへの戻り値を格納

begin
	$exit_code = main

	notice_message = "正常終了\n" + make_notice_msg
	$logger.info notice_message

rescue => ex
	if $exit_code == EXIT_CODE::NORMAL_END
		$exit_code = EXIT_CODE::GENERAL_FAILURE # 終了コードが正常を表したままなら、異常を表す値に変える
	end

	# アップロードしたJSONファイルの元々のURLをクリアして、次回は確実にJSONをダウンロードさせる
	clear_json_url $msdc_json_url_file

	error_message = make_error_msg_from_exception ex, "〓〓〓〓〓 例外が発生しました（#{$my_own_title}） 〓〓〓〓〓"
	$logger.fatal error_message

	if !$debug # デバッグ時は通知しない。うざいので
		success_sending_mail = send_alert_by_mail "【死活監視】例外が発生しました", error_message
		$logger.error "メール送信に失敗しました" if !success_sending_mail
	end

ensure
	$logger.close if $logger

end

exit $exit_code
